const closeAssignModal = () => {
    const modal = document.getElementById('assignModal');
    if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('active');
    }
};

function loadAssistants(page = 1) {
    const tbody = document.getElementById('assign-tbody');
    if (!tbody) return;

    tbody.innerHTML = '<tr><td colspan="4" class="text-center">Loading...</td></tr>';

    // First, get all vector stores to have their names
    fetch(`api/VectorStoresController.php?action=list`)
        .then(response => response.json())
        .then(vectorStoresData => {
            const vectorStores = vectorStoresData.data || [];
            
            // Then get assistants
            return fetch(`api/AssistantsController.php?action=list&page=${page}`)
                .then(response => response.json())
                .then(data => {
                    tbody.innerHTML = '';
                    
                    if (!data.data || data.data.length === 0) {
                        tbody.innerHTML = '<tr><td colspan="4" class="text-center">No assistants found</td></tr>';
                        return;
                    }
                    
                    data.data.forEach(assistant => {
                        // Get assigned vector store IDs
                        const assignedIds = assistant.toolResources?.fileSearch?.vectorStoreIds || [];
                        const assignedStores = assignedIds.map(id => {
                            const store = vectorStores.find(s => s.id === id);
                            return store ? `${store.name} (${id})` : id;
                        }).join(', ');

                        tbody.innerHTML += `
                            <tr>
                                <td>
                                    <div>${assistant.name || 'Unnamed'}</div>
                                    <div class="assistant-id">${assistant.id}</div>
                                </td>
                                <td>${assistant.model}</td>
                                <td>
                                    <div class="assigned-stores">
                                        ${assignedStores || '<em>No vector stores assigned</em>'}
                                    </div>
                                </td>
                                <td>
                                    <button onclick="showAssignModal('${assistant.id}')" class="btn-icon" title="Assign Vector Stores">
                                        <i class="material-icons">link</i>
                                    </button>
                                </td>
                            </tr>
                        `;
                    });
                });
        })
        .catch(error => {
            console.error('Error:', error);
            tbody.innerHTML = `
                <tr>
                    <td colspan="4" class="text-center text-error">
                        Error loading assistants: ${error.message}
                    </td>
                </tr>
            `;
        });
}

function showAssignModal(assistantId) {
    const modal = document.getElementById('assignModal');
    const form = document.getElementById('assignForm');
    const vectorStoresList = document.getElementById('vectorStoresList');
    
    document.getElementById('assistant_id').value = assistantId;
    vectorStoresList.innerHTML = '<div class="loading">Loading vector stores...</div>';
    
    modal.style.display = 'flex';
    modal.classList.add('active');

    // Load current assignments and available vector stores
    Promise.all([
        fetch(`api/AssistantsController.php?action=retrieve&id=${assistantId}`).then(r => r.json()),
        fetch(`api/VectorStoresController.php?action=list`).then(r => r.json())
    ])
    .then(([assistantData, vectorStores]) => {
        const assignedIds = assistantData.data?.toolResources?.fileSearch?.vectorStoreIds || [];
        const stores = vectorStores.data || [];
        
        // Sort stores to put assigned ones at the top
        const sortedStores = [...stores].sort((a, b) => {
            const aAssigned = assignedIds.includes(a.id);
            const bAssigned = assignedIds.includes(b.id);
            if (aAssigned && !bAssigned) return -1;
            if (!aAssigned && bAssigned) return 1;
            return 0;
        });
        
        vectorStoresList.innerHTML = `
            <div class="note-warning">Only one vector store can be assigned to an assistant.</div>
            ${sortedStores.map(store => `
                <label class="checkbox-item ${assignedIds.includes(store.id) ? 'assigned' : ''}">
                    <input type="radio" 
                           name="vector_stores[]" 
                           value="${store.id}"
                           ${assignedIds.includes(store.id) ? 'checked' : ''}>
                    <span class="checkbox-label">
                        <strong>${store.name}</strong>
                        <small class="store-id">${store.id}</small>
                    </span>
                </label>
            `).join('') || '<div class="text-center">No vector stores available</div>'}
        `;
    })
    .catch(error => {
        console.error('Error:', error);
        vectorStoresList.innerHTML = `<div class="text-error">Error loading vector stores: ${error.message}</div>`;
    });
}

function saveAssignments(event) {
    event.preventDefault();
    
    const form = event.target;
    const submitButton = form.querySelector('button[type="submit"]');
    showLoader(submitButton);
    
    const assistantId = document.getElementById('assistant_id').value;
    const vectorStoreIds = Array.from(form.querySelectorAll('input[name="vector_stores[]"]:checked'))
        .map(input => input.value);

    console.log('Sending assignment request:', {
        assistant_id: assistantId,
        vector_store_ids: vectorStoreIds
    });

    fetch('api/AssignController.php?action=assign', {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify({
            assistant_id: assistantId,
            vector_store_ids: vectorStoreIds
        })
    })
    .then(response => response.json())
    .then(data => {
        console.log('Assignment response:', data);
        if (data.success) {
            showNotification('Vector Store assigned successfully');
            closeAssignModal();
            loadAssistants(currentPage);
        } else {
            showNotification('Error assigning vector store', 'error');
        }
        hideLoader(submitButton);
    })
    .catch(error => {
        console.error('Error:', error);
        showNotification(error.message, 'error');
        hideLoader(submitButton);
    });
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', () => {
    if (window.location.href.includes('page=assign')) {
        loadAssistants();
    }
}); 