function loadAssistantsList() {
    console.log('loadAssistantsList');
    var page = 1;
    try {
        console.log('=== loadAssistantsList START ===');
        console.log('Page:', page);
        
        const tbody = document.getElementById('assistants-tbody');
        if (!tbody) {
            throw new Error('Assistants tbody element not found');
        }

        console.log('Found tbody, setting loading message');
        tbody.innerHTML = '<tr><td colspan="3" class="text-center">Loading...</td></tr>';
   
        const timestamp = new Date().getTime();
        const url = `api/AssistantsController.php?action=list&page=${page}&_=${timestamp}`;
        console.log('Fetching from URL:', url);

        // Add debug for fetch start
        console.log('Starting fetch request...');
        
        fetch(url, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
                'X-Requested-With': 'XMLHttpRequest'
            }
        })
        .then(response => {
            console.log('Response received:', response);
            console.log('Response status:', response.status);
            console.log('Response headers:', Array.from(response.headers.entries()));
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            return response.json();
        })
        .then(data => {
            console.log('Data received:', data);
            tbody.innerHTML = '';
            
            if (!data.data || data.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="3" class="text-center">No assistants found</td></tr>';
                return;
            }
            
            data.data.forEach(assistant => {
                tbody.innerHTML += `
                    <tr>
                        <td>
                            <div>${assistant.name || 'Unnamed'}</div>
                            <div class="assistant-id">${assistant.id}</div>
                        </td>
                        <td>${assistant.model}</td>
                        <td>
                            <button onclick="editAssistant('${assistant.id}')" class="btn-icon" title="Edit">
                                <i class="material-icons">edit</i>
                            </button>
                            <button onclick="deleteAssistant('${assistant.id}')" class="btn-icon delete" title="Delete">
                                <i class="material-icons">delete</i>
                            </button>
                        </td>
                    </tr>
                `;
            });
        })
        .catch(error => {
            console.error('Fetch error:', error);
            console.error('Error stack:', error.stack);
            tbody.innerHTML = `
                <tr>
                    <td colspan="3" class="text-center text-error">
                        Error loading assistants: ${error.message}
                    </td>
                </tr>
            `;
        });
    } catch (error) {
        console.error('loadAssistantsList error:', error);
        console.error('Error stack:', error.stack);
    }
}

function createAssistant() {
    const modal = document.getElementById('assistantModal');
    const form = document.getElementById('assistantForm');
    
    // Reset form
    form.innerHTML = `
        <input type="hidden" id="assistant_id" name="assistant_id">
        
        <div class="form-group">
            <label for="name">Name</label>
            <input type="text" id="name" name="name" required>
        </div>

        <div class="form-group">
            <label for="instructions">Instructions</label>
            <textarea id="instructions" name="instructions" required></textarea>
        </div>

        <div class="form-group">
            <label for="model">Model</label>
            <select id="model" name="model" required>
                <option value="gpt-4">GPT-4</option>
                <option value="gpt-4-turbo-preview">GPT-4 Turbo</option>
                <option value="gpt-4o-mini">GPT-4O Mini</option>
                <option value="gpt-4-32k">GPT-4 32K</option>
                <option value="gpt-3.5-turbo">GPT-3.5 Turbo</option>
                <option value="gpt-3.5-turbo-16k">GPT-3.5 Turbo 16K</option>
            </select>
        </div>

        <div class="form-actions">
            <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
            <button type="submit" class="btn btn-primary" id="submitButton">Create</button>
        </div>
    `;
    
    form.reset();
    document.getElementById('assistant_id').value = '';
    document.getElementById('modalTitle').textContent = 'Create Assistant';
    
    // Add form submit handler
    form.onsubmit = saveAssistant;
    
    // Show modal
    modal.style.display = 'block';
}

function editAssistant(assistantId) {
    const modal = document.getElementById('assistantModal');
    const form = document.getElementById('assistantForm');
    document.getElementById('modalTitle').textContent = 'Edit Assistant';
    
    // Show loading state
    form.innerHTML = '<div class="loading">Loading...</div>';
    modal.style.display = 'block';
    
    fetch(`api/AssistantsController.php?action=retrieve&id=${assistantId}`)
        .then(response => {
            if (!response.ok) throw new Error('Network response was not ok');
            return response.json();
        })
        .then(data => {
            if (data.error) throw new Error(data.error);
            
            // Restore form HTML with the same structure as createAssistant
            form.innerHTML = `
                <input type="hidden" id="assistant_id" name="assistant_id">
                
                <div class="form-group">
                    <label for="name">Name</label>
                    <input type="text" id="name" name="name" required>
                </div>

                <div class="form-group">
                    <label for="instructions">Instructions</label>
                    <textarea id="instructions" name="instructions" required></textarea>
                </div>

                <div class="form-group">
                    <label for="model">Model</label>
                    <select id="model" name="model" required>
                        <option value="gpt-4">GPT-4</option>
                        <option value="gpt-4-turbo-preview">GPT-4 Turbo</option>
                        <option value="gpt-4o-mini">GPT-4O Mini</option>
                        <option value="gpt-4-32k">GPT-4 32K</option>
                        <option value="gpt-3.5-turbo">GPT-3.5 Turbo</option>
                        <option value="gpt-3.5-turbo-16k">GPT-3.5 Turbo 16K</option>
                    </select>
                </div>

                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary" id="submitButton">Save Changes</button>
                </div>
            `;
            
            // Set form values
            const assistant = data.data;
            document.getElementById('assistant_id').value = assistant.id;
            document.getElementById('name').value = assistant.name || '';
            document.getElementById('instructions').value = assistant.instructions || '';
            document.getElementById('model').value = assistant.model;
            
            // Add form submit handler
            form.onsubmit = saveAssistant;
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Error loading assistant: ' + error.message, 'error');
            closeModal();
        });
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', () => {
    console.log('DOM Content Loaded');
    console.log('Current URL:', window.location.href);
    
    const currentUrl = window.location.href;
    const urlParams = new URLSearchParams(window.location.search);
    const currentPage = urlParams.get('page');
    
    console.log('Current page parameter:', currentPage);
    
    const isAssistantsPage = !currentPage || currentPage === 'assistants';
    
    console.log('Is assistants page:', isAssistantsPage);
    
    if (isAssistantsPage) {
        console.log('Should load assistants');
        const tbody = document.getElementById('assistants-tbody');
        
        if (!tbody) {
            console.error('Assistants tbody element not found');
            console.log('Current HTML:', document.body.innerHTML);
            return;
        }
        
        console.log('Found tbody, calling loadAssistantsList()');
        loadAssistantsList();
    } else {
        console.log('Not on assistants page, skipping load');
    }
});



function updatePagination(hasMore, currentPage) {
    const pagination = document.getElementById('assistants-pagination');
    let html = '';
    
    if (currentPage > 1) {
        html += `<button onclick="loadAssistantsList(${currentPage - 1})">Previous</button>`;
    }
    
    html += `<button class="active">${currentPage}</button>`;
    
    if (hasMore) {
        html += `<button onclick="loadAssistantsList(${currentPage + 1})">Next</button>`;
    }
    
    pagination.innerHTML = html;
}

function saveAssistant(event) {
    event.preventDefault();
    
    const saveButton = document.getElementById('submitButton');
    if (!saveButton) {
        console.error('Submit button not found');
        return;
    }
    
    // Just disable the button
    saveButton.disabled = true;
    
    const formData = new FormData(document.getElementById('assistantForm'));
    const assistantId = formData.get('assistant_id');
    
    const data = {
        name: formData.get('name'),
        instructions: formData.get('instructions'),
        model: formData.get('model'),
        tools: ["file_search"],
        metadata: {},
        file_ids: Array.from(formData.getAll('file_ids[]')) || []
    };

    if (assistantId) {
        data.assistant_id = assistantId;
    }

    console.log('Sending data:', JSON.stringify(data, null, 2));

    fetch(`api/AssistantsController.php?action=${assistantId ? 'update' : 'create'}`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(async response => {
        const responseData = await response.json();
        console.log('Response:', responseData);
        if (!response.ok) {
            throw new Error(responseData.error || 'Failed to save assistant');
        }
        return responseData;
    })
    .then(data => {
        if (data.success) {
            closeModal();
            loadAssistantsList();
            showNotification('Assistant saved successfully');
        } else {
            throw new Error(data.error || 'Failed to save assistant');
        }
    })
    .catch(error => {
        console.error('Save error:', error);
        showNotification('Error saving assistant: ' + error.message, 'error');
    })
    .finally(() => {
        // Just re-enable the button
        saveButton.disabled = false;
    });
}

function deleteAssistant(assistantId) {
    if (confirm('Are you sure you want to delete this assistant?')) {
        fetch('api/AssistantsController.php?action=delete', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ assistant_id: assistantId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                loadAssistantsList();
                showNotification('Assistant deleted successfully');
            } else {
                showNotification('Error deleting assistant', 'error');
            }
        });
    }
}

// Modal functions
function openModal() {
    document.getElementById('assistantModal').style.display = 'block';
}

function closeModal() {
    document.getElementById('assistantModal').style.display = 'none';
}

// Notification function
function showNotification(message, type = 'success') {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.textContent = message;
    document.body.appendChild(notification);
    
    setTimeout(() => {
        notification.remove();
    }, 3000);
} 