let isProcessing = false;
let activeThreadId = null;

async function loadVectorStores() {
    try {
        const response = await fetch('/assistant/api/VectorStoresController.php?action=list&page=1');
        const data = await response.json();
        
        const select = document.getElementById('vectorStoreSelect');
        select.innerHTML = '<option value="">Select Vector Store</option>';
        
        if (data.success && data.data) {
            data.data.forEach(store => {
                select.innerHTML += `
                    <option value="${store.id}">${store.name}</option>
                `;
            });
            
            // If there's only one vector store, select it automatically
            if (data.data.length === 1) {
                select.value = data.data[0].id;
            }
        }
    } catch (error) {
        console.error('Error loading vector stores:', error);
        showNotification('Failed to load vector stores', 'error');
    }
}

function loadThreads() {
    const threads = JSON.parse(localStorage.getItem('chatThreads') || '{}');
    const threadsList = document.getElementById('threadsList');
    threadsList.innerHTML = '';
    
    Object.entries(threads)
        .sort((a, b) => new Date(b[1].created) - new Date(a[1].created))
        .forEach(([threadId, threadData]) => {
            const threadEl = document.createElement('div');
            threadEl.className = `thread-item ${threadId === activeThreadId ? 'active' : ''}`;
            threadEl.onclick = () => loadThread(threadId);
            threadEl.innerHTML = `
                <i class="material-icons">chat</i>
                <div class="thread-name">${threadData.name || 'Untitled Chat'}</div>
            `;
            threadsList.appendChild(threadEl);
        });
}

async function createNewThread() {
    if (isProcessing) return;
    
    const vectorStoreId = document.getElementById('vectorStoreSelect').value;
    if (!vectorStoreId) {
        showNotification('Please select a vector store', 'error');
        return;
    }
    
    isProcessing = true;
    
    try {
        const response = await fetch(`/assistant/api/ChatController.php?vectorStoreId=${vectorStoreId}`);
        const data = await response.json();
        
        if (data.success) {
            const threadId = data.thread_id;
            const threads = JSON.parse(localStorage.getItem('chatThreads') || '{}');
            
            // Add new thread
            threads[threadId] = {
                name: 'New Chat',
                created: new Date().toISOString(),
                vectorStoreId: vectorStoreId,
                messages: [] // Initialize empty messages array
            };
            
            localStorage.setItem('chatThreads', JSON.stringify(threads));
            
            // Clear existing chat and load new thread
            document.getElementById('chatMessages').innerHTML = '';
            document.getElementById('threadName').value = 'New Chat';
            activeThreadId = threadId;
            
            loadThreads();
            showNotification('New chat created');
        }
    } catch (error) {
        showNotification('Failed to create thread', 'error');
    } finally {
        isProcessing = false;
    }
}

async function loadThread(threadId) {
    if (isProcessing) return;
    isProcessing = true;
    
    try {
        activeThreadId = threadId;
        const threads = JSON.parse(localStorage.getItem('chatThreads') || '{}');
        const threadData = threads[threadId];
        
        if (!threadData) {
            showNotification('Thread not found', 'error');
            return;
        }
        
        // Update UI
        document.getElementById('threadName').value = threadData.name || '';
        document.getElementById('vectorStoreSelect').value = threadData.vectorStoreId || '';
        
        // Load messages from API
        await retrieveThreadMessages(threadId);
        
        loadThreads(); // Refresh thread list to show active state
    } catch (error) {
        console.error('Error loading thread:', error);
        showNotification('Failed to load thread', 'error');
    } finally {
        isProcessing = false;
    }
}

function updateThreadName() {
    if (!activeThreadId) return;
    
    const newName = document.getElementById('threadName').value;
    const threads = JSON.parse(localStorage.getItem('chatThreads') || '{}');
    
    if (threads[activeThreadId]) {
        threads[activeThreadId].name = newName;
        localStorage.setItem('chatThreads', JSON.stringify(threads));
        loadThreads();
    }
}

async function deleteThread() {
    if (!activeThreadId || isProcessing) return;
    
    if (!confirm('Are you sure you want to delete this chat?')) return;
    
    isProcessing = true;
    
    try {
        const response = await fetch('/assistant/api/ChatController.php', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                action: 'delete',
                threadId: activeThreadId
            })
        });
        
        const data = await response.json();
        
        if (data.success && data.deleted) {
            // Remove from localStorage
            const threads = JSON.parse(localStorage.getItem('chatThreads') || '{}');
            delete threads[activeThreadId];
            localStorage.setItem('chatThreads', JSON.stringify(threads));
            
            // Clear UI
            document.getElementById('chatMessages').innerHTML = '';
            document.getElementById('threadName').value = '';
            activeThreadId = null;
            
            // Refresh thread list
            loadThreads();
            showNotification('Chat deleted successfully');
        } else {
            throw new Error(data.error || 'Failed to delete chat from OpenAI');
        }
    } catch (error) {
        console.error('Delete error:', error);
        showNotification('Failed to delete chat: ' + error.message, 'error');
    } finally {
        isProcessing = false;
    }
}

async function retrieveThreadMessages(threadId) {
    try {
        const response = await fetch(`/assistant/api/ChatController.php?action=retrieve&threadId=${threadId}`);
        const data = await response.json();
        
        if (data.success && data.messages) {
            const messagesDiv = document.getElementById('chatMessages');
            messagesDiv.innerHTML = '';
            
            data.messages.data.forEach(msg => {
                const content = msg.content[0].text.value;
                
                // Get unique sources if they exist
                let sources = null;
                if (msg.sources) {
                    // Sources is already an array, just ensure uniqueness
                    sources = [...new Set(msg.sources)];
                }
                
                // Get citations if they exist
                let citations = null;
                if (msg.citations) {
                    citations = msg.citations.map(citation => ({
                        ...citation,
                        file_name: citation.file_name.replace('.txt', '').replace(/_/g, ' ')
                    }));
                }
                
                addMessage(content, msg.role, sources, citations);
            });
        }
    } catch (error) {
        console.error('Error retrieving messages:', error);
        showNotification('Failed to load messages', 'error');
    }
}

// Message handling
async function sendMessage() {
    if (!activeThreadId || isProcessing) return;
    
    const messageInput = document.getElementById('messageInput');
    const message = messageInput.value.trim();
    if (!message) return;
    
    isProcessing = true;
    messageInput.value = '';
    
    addMessage(message, 'user');
    
    const useStream = document.getElementById('streamToggle').checked;
    let url = `/assistant/api/ChatController.php?threadId=${activeThreadId}&messages=${encodeURIComponent(message)}`;
    
    if (useStream) {
        handleStreamResponse(url + '&stream=true');
    } else {
        handleRegularResponse(url);
    }
}

function splitCodeFromText(text) {
    const regex = /```(\w+)?\s*([\s\S]+?)```|([\s\S]+?(?=```|$))/g;
    const parts = [];
    let lastIndex = 0;

    while (true) {
        const match = regex.exec(text);
        if (!match) break;

        if (match[2]) {
            if (match.index > lastIndex) {
                const nonCodeText = text.substring(lastIndex, match.index);
                parts.push({ isCode: false, text: nonCodeText });
            }
            parts.push({ isCode: true, text: match[2], language: match[1] || undefined });
            lastIndex = regex.lastIndex;
        } else if (match[3]) {
            parts.push({ isCode: false, text: match[3] });
            lastIndex = regex.lastIndex;
        }
    }

    return parts;
}

function formatMessageContent(content) {
    const parts = splitCodeFromText(content);
    return parts.map(part => {
        if (part.isCode) {
            return `<pre><code class="language-${part.language || ''}">${part.text}</code></pre>`;
        } else {
            // Apply existing formatting for regular text
            return part.text
                .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
                .replace(/(\d+\.\s+)/g, '<br>$1')
                .replace(/\.\s{2,}/g, '.<br>')
                .replace(/\.\s+(?=[A-Z])/g, '.<br>')
                .replace(/،\s+/g, '،<br>')
                .replace(/\.\s+(?=[\u0600-\u06FF])/g, '.<br>');
        }
    }).join('');
}

function addMessage(content, role, sources = null, citations = null) {
    const messagesDiv = document.getElementById('chatMessages');
    const messageDiv = document.createElement('div');
    messageDiv.className = `message ${role}`;
    
    const isArabic = /[\u0600-\u06FF]/.test(content);
    const formattedContent = formatMessageContent(content);

    let messageHtml = `
        <div class="message-content" dir="${isArabic ? 'rtl' : 'ltr'}" 
             style="text-align: ${isArabic ? 'right' : 'left'}">
            ${formattedContent}
        </div>
    `;

    // Add sources and citations if they exist
    if (sources && sources.length > 0) {
        messageHtml += `
            <div class="message-sources" dir="${isArabic ? 'rtl' : 'ltr'}" 
                 style="text-align: ${isArabic ? 'right' : 'left'}">
                <hr>
                <strong>${isArabic ? 'المصادر:' : 'Sources:'}</strong>
                <ul>
        `;

        // Only show unique sources in the list
        const uniqueSources = [...new Set(sources)];
        uniqueSources.forEach(source => {
            messageHtml += `<li>${source}</li>`;
        });

        messageHtml += `</ul></div>`;
    }

    messageDiv.innerHTML = messageHtml;
    messagesDiv.appendChild(messageDiv);
    messagesDiv.scrollTop = messagesDiv.scrollHeight;
}

// Typing indicator functions
function showTypingIndicator() {
    const messagesDiv = document.getElementById('chatMessages');
    const typingDiv = document.createElement('div');
    typingDiv.className = 'message assistant';
    typingDiv.id = 'typingIndicator';
    typingDiv.innerHTML = `
        <div class="typing-indicator">
            <span></span>
            <span></span>
            <span></span>
        </div>
    `;
    messagesDiv.appendChild(typingDiv);
    messagesDiv.scrollTop = messagesDiv.scrollHeight;
}

function hideTypingIndicator() {
    const typingDiv = document.getElementById('typingIndicator');
    if (typingDiv) {
        typingDiv.remove();
    }
}

async function handleRegularResponse(url) {
    try {
        showTypingIndicator();
        const response = await fetch(url);
        const data = await response.json();
        
        console.log('Server response:', data.replay);
        
        hideTypingIndicator();
        
        if (data.success && data.replay) {
            // Get the message content
            const content = data.replay.content[0].text.value;
            
            // Get sources and citations from the replay
            let sources = [];
            let citations = [];
            
            // Check if sources exist in the replay
            if (data.replay.sources) {
                // Sources is already an array, just ensure uniqueness
                sources = Array.isArray(data.replay.sources) 
                    ? [...new Set(data.replay.sources)]
                    : [...new Set(Object.values(data.replay.sources))];
            }
            
            // Get citations if they exist
            if (data.replay.citations) {
                citations = data.replay.citations.map(citation => ({
                    ...citation,
                    file_name: citation.file_name.replace('.txt', '').replace(/_/g, ' ')
                }));
            }
            
            // Add the message with sources and citations
            addMessage(
                content, 
                'assistant',
                sources.length > 0 ? sources : null,
                citations.length > 0 ? citations : null
            );
        }
    } catch (error) {
        hideTypingIndicator();
        showNotification('Failed to send message', 'error');
        console.error('Error:', error);
    } finally {
        isProcessing = false;
    }
}

function handleStreamResponse(url) {
    let messageContent = '';
    let messageDiv = null;
    let sources = null;
    let citations = null;
    
    showTypingIndicator();
    
    fetch(url)
        .then(response => {
            const reader = response.body.getReader();
            const decoder = new TextDecoder();
            
            return new ReadableStream({
                start(controller) {
                    function push() {
                        reader.read().then(({done, value}) => {
                            if (done) {
                                hideTypingIndicator();
                                // Format the final message when stream is complete
                                if (messageDiv) {
                                    messageDiv.querySelector('.message-content').innerHTML = 
                                        formatMessageContent(messageContent);
                                    // Add sources if we received them
                                    if (sources) {
                                        addSourcesToMessage(messageDiv, sources, citations);
                                    }
                                }
                                controller.close();
                                isProcessing = false;
                                return;
                            }
                            
                            const chunk = decoder.decode(value, {stream: true});
                            const lines = chunk.split('\n\n');
                            
                            lines.forEach(line => {
                                if (line.trim()) {
                                    try {
                                        const data = JSON.parse(line);
                                        if (data.isSourceInfo) {
                                            // Store sources for later use
                                            sources = data.sources;
                                            citations = data.citations;
                                        } else if (data.content) {
                                            if (!messageDiv) {
                                                hideTypingIndicator();
                                                messageDiv = document.createElement('div');
                                                messageDiv.className = 'message assistant';
                                                messageDiv.innerHTML = `
                                                    <div class="message-content"></div>
                                                `;
                                                document.getElementById('chatMessages').appendChild(messageDiv);
                                            }
                                            messageContent += data.content + ' ';
                                            messageDiv.querySelector('.message-content').textContent = messageContent;
                                            messageDiv.scrollIntoView({behavior: 'smooth'});
                                        }
                                    } catch (e) {
                                        console.error('Error parsing stream chunk:', e);
                                    }
                                }
                            });
                            
                            controller.enqueue(value);
                            push();
                        });
                    }
                    
                    push();
                }
            });
        })
        .catch(error => {
            hideTypingIndicator();
            showNotification('Error in stream response', 'error');
            isProcessing = false;
        });
}

// Add this helper function to add sources to a message
function addSourcesToMessage(messageDiv, sources, citations) {
    const isArabic = /[\u0600-\u06FF]/.test(messageDiv.textContent);
    let sourcesHtml = `
        <div class="message-sources" dir="${isArabic ? 'rtl' : 'ltr'}" 
             style="text-align: ${isArabic ? 'right' : 'left'}">
            <hr>
            <strong>${isArabic ? 'المصادر:' : 'Sources:'}</strong>
            <ul>
    `;

    if (citations && citations.length > 0) {
        citations.forEach(citation => {
            sourcesHtml += `
                <li>
                    <strong>${citation.file_name}</strong>
                    ${citation.quote ? `<blockquote>${citation.quote}</blockquote>` : ''}
                </li>
            `;
        });
    } else {
        sources.forEach(source => {
            sourcesHtml += `<li>${source}</li>`;
        });
    }

    sourcesHtml += `</ul></div>`;
    messageDiv.insertAdjacentHTML('beforeend', sourcesHtml);
}

function handleKeyPress(event) {
    if (event.key === 'Enter' && !event.shiftKey) {
        event.preventDefault();
        sendMessage();
    }
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', () => {
    loadVectorStores();
    loadThreads();
}); 