let currentPage = 1;
const itemsPerPage = 10;
const BATCH_SIZE = 10; // Number of files to upload at once

function loadFiles(page = 1) {
    currentPage = page;
    
    // Show loading state
    document.getElementById('loadingState').style.display = 'flex';
    document.getElementById('emptyState').style.display = 'none';
    document.getElementById('errorState').style.display = 'none';
    document.getElementById('filesList').style.display = 'none';

    fetch(`api/FilesController.php?action=list&page=${page}&limit=${itemsPerPage}`)
        .then(response => response.json())
        .then(data => {
            document.getElementById('loadingState').style.display = 'none';
            
            if (!data.data || data.data.length === 0) {
                document.getElementById('emptyState').style.display = 'flex';
                return;
            }

            const filesContainer = document.getElementById('filesList');
            filesContainer.style.display = 'block';
            filesContainer.innerHTML = `
                <div class="files-list-content">
                    ${data.data.map(file => `
                        <div class="file-item">
                            <div class="file-name" onclick="showFileDetails('${file.id}')">
                                <i class="material-icons">description</i>
                                ${file.filename}
                            </div>
                            <div class="file-size">${formatBytes(file.bytes)}</div>
                            <div class="file-status ${file.status}">${file.status}</div>
                            <div class="file-actions">
                                <button onclick="deleteFile(event, '${file.id}')" 
                                        class="btn-icon delete" title="Delete File">
                                    <i class="material-icons">delete</i>
                                </button>
                            </div>
                        </div>
                    `).join('')}
                </div>
            `;
        })
        .catch(error => {
            document.getElementById('loadingState').style.display = 'none';
            document.getElementById('errorState').style.display = 'flex';
            document.getElementById('errorState').querySelector('.message').textContent = 
                'Error loading files: ' + error.message;
        });
}

function showFileDetails(fileId) {
    const modal = document.getElementById('fileDetailsModal');
    const detailsContainer = document.getElementById('fileDetails');
    
    modal.style.display = 'flex';
    modal.classList.add('active');
    
    detailsContainer.innerHTML = `
        <div class="center-loader">
            <div class="spinner"></div>
            <div class="text">Loading file details...</div>
        </div>
    `;

    fetch(`api/FilesController.php?action=retrieve&id=${fileId}`)
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                const file = data.data;
                detailsContainer.innerHTML = `
                    <div class="file-details-grid">
                        <div class="detail-row">
                            <span class="label">File ID:</span>
                            <span class="value">${file.id}</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Filename:</span>
                            <span class="value">${file.filename}</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Size:</span>
                            <span class="value">${formatBytes(file.bytes)}</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Created:</span>
                            <span class="value">${formatDate(file.created_at)}</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Purpose:</span>
                            <span class="value">${file.purpose}</span>
                        </div>
                        <div class="detail-row">
                            <span class="label">Status:</span>
                            <span class="value ${file.status}">${file.status}</span>
                        </div>
                    </div>
                `;
            }
        });
}

function closeFileDetailsModal() {
    const modal = document.getElementById('fileDetailsModal');
    modal.style.display = 'none';
    modal.classList.remove('active');
}

function deleteFile(event, fileId) {
    event.preventDefault();
    event.stopPropagation();

    if (confirm('Are you sure you want to delete this file?')) {
        const button = event.currentTarget;
        const fileItem = button.closest('.file-item'); // Get the parent file item
        const originalContent = button.innerHTML;
        button.innerHTML = '<div class="loader"></div>';
        button.disabled = true;

        fetch('api/FilesController.php?action=delete', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                file_id: fileId
            })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('File deleted successfully');
                
                // Animate and remove the file item
                fileItem.style.transition = 'all 0.3s ease';
                fileItem.style.opacity = '0';
                fileItem.style.height = '0';
                fileItem.style.padding = '0';
                fileItem.style.margin = '0';
                
                setTimeout(() => {
                    fileItem.remove();
                    
                    // Check if there are no more files
                    const filesList = document.getElementById('filesList');
                    if (!filesList.querySelector('.file-item')) {
                        document.getElementById('filesList').style.display = 'none';
                        document.getElementById('emptyState').style.display = 'flex';
                    }
                }, 300);
            } else {
                showNotification(data.error || 'Error deleting file', 'error');
                button.innerHTML = originalContent;
                button.disabled = false;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification(error.message, 'error');
            button.innerHTML = originalContent;
            button.disabled = false;
        });
    }
}

function deleteAllFiles() {
    if (confirm('Are you sure you want to delete all files? This action cannot be undone.')) {
        const button = event.target.closest('button');
        button.disabled = true;
        button.innerHTML = '<div class="loader"></div>';

        fetch('api/FilesController.php?action=deleteAll', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            }
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification(`${data.message || 'All files deleted successfully'}`);
                loadFiles(); // Reload the files list
            } else {
                showNotification(data.error || 'Error deleting files', 'error');
            }
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to delete files', 'error');
        })
        .finally(() => {
            button.disabled = false;
            button.innerHTML = '<i class="material-icons">delete_sweep</i> Delete All Files';
        });
    }
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', () => {
    if (window.location.href.includes('page=files')) {
        loadFiles();
    }
});

function formatBytes(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

function formatDate(timestamp) {
    const date = new Date(timestamp * 1000);
    const options = {
        year: 'numeric',
        month: 'short',
        day: 'numeric',
        hour: '2-digit',
        minute: '2-digit',
        hour12: true
    };
    return date.toLocaleString('en-US', options);
}

async function uploadFiles(event) {
    event.preventDefault();
    const form = event.target;
    const fileInput = form.querySelector('input[type="file"]');
    const files = Array.from(fileInput.files);
    const totalFiles = files.length;
    
    if (totalFiles === 0) return;

    const submitButton = form.querySelector('button[type="submit"]');
    submitButton.disabled = true;
    
    let completedFiles = 0;
    let failedFiles = 0;
    
    // Process files in batches
    const BATCH_SIZE = 10;
    for (let i = 0; i < files.length; i += BATCH_SIZE) {
        const batch = files.slice(i, i + BATCH_SIZE);
        const batchPromises = batch.map(file => {
            const formData = new FormData();
            formData.append('file', file);
            
            return fetch('api/FilesController.php?action=upload', {
                method: 'POST',
                body: formData
            })
            .then(response => response.json())
            .then(data => {
                completedFiles++;
                updateProgress(completedFiles, totalFiles);
                
                if (!data.success) {
                    failedFiles++;
                    console.error(`Failed to upload ${file.name}:`, data.error);
                }
            })
            .catch(error => {
                completedFiles++;
                failedFiles++;
                updateProgress(completedFiles, totalFiles);
                console.error(`Error uploading ${file.name}:`, error);
            });
        });

        // Wait for current batch to complete before starting next batch
        await Promise.all(batchPromises);
    }

    // All files completed
    submitButton.disabled = false;
    if (failedFiles > 0) {
        showNotification(`Upload completed with ${failedFiles} failures`, 'error');
    } else {
        showNotification('All files uploaded successfully');
    }
    loadFiles();
    form.reset();
}

function updateProgress(completed, total) {
    const submitButton = document.querySelector('.upload-submit');
    const percent = Math.round((completed / total) * 100);
    submitButton.innerHTML = `<i class="material-icons">cloud_upload</i> Uploading... ${percent}%`;
} 