const closeFilesModal = () => {
    const modal = document.getElementById('vectorStoreFilesModal');
    if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('active');
        
        // Reset the file input
        const fileInput = modal.querySelector('input[type="file"]');
        if (fileInput) {
            fileInput.value = '';
        }
        
        // Clear the files list
        const filesContainer = document.getElementById('filesToUpload');
        if (filesContainer) {
            filesContainer.innerHTML = '<div class="no-files">No files selected</div>';
        }
        
        // Reset the upload button
        const submitButton = modal.querySelector('.submit-upload-btn');
        if (submitButton) {
            submitButton.disabled = true;
            submitButton.innerHTML = '<i class="material-icons">cloud_upload</i> Upload Files';
        }
    }
};

function showFilesModal(vectorStoreId) {
    const modal = document.getElementById('vectorStoreFilesModal');
    document.getElementById('current_vectorstore_id').value = vectorStoreId;
    
    // Clear the files list when opening
    const filesContainer = document.getElementById('filesToUpload');
    if (filesContainer) {
        filesContainer.innerHTML = '<div class="no-files">No files selected</div>';
    }
    
    // Reset the file input
    const fileInput = modal.querySelector('input[type="file"]');
    if (fileInput) {
        fileInput.value = '';
    }
    
    // Reset the upload button
    const submitButton = modal.querySelector('.submit-upload-btn');
    if (submitButton) {
        submitButton.disabled = true;
        submitButton.innerHTML = '<i class="material-icons">cloud_upload</i> Upload Files';
    }
    
    modal.style.display = 'flex';
    modal.classList.add('active');
    
    loadVectorStoreFiles(vectorStoreId);
}

function loadVectorStoreFiles(vectorStoreId, after = null) {
    const filesContainer = document.getElementById('vectorStoreFiles');
    filesContainer.innerHTML = `
        <div class="loading">
            <div class="spinner"></div>
            <div>Loading files...</div>
        </div>
    `;

    const url = `api/VectorStoresController.php?action=listFiles&vectorstore_id=${vectorStoreId}${after ? `&after=${after}` : ''}`;
    
    fetch(url)
        .then(response => response.json())
        .then(data => {
            if (!data.data || data.data.length === 0) {
                filesContainer.innerHTML = `
                    <div class="loading">
                        <i class="material-icons" style="font-size: 48px; color: var(--secondary-color);">folder_open</i>
                        <div>No files uploaded yet</div>
                    </div>
                `;
                return;
            }

            filesContainer.innerHTML = `
                <div class="files-list-content">
                    ${data.data.map(file => `
                        <div class="file-item">
                            <div class="file-name">
                                <i class="material-icons">description</i>
                                ${file.filename}
                            </div>
                            <div class="file-size">${formatFileSize(file.size)}</div>
                            <div class="file-date">${formatDate(file.created_at)}</div>
                            <div class="file-status ${file.status}">${file.status}</div>
                            <div class="file-actions">
                                <button onclick="deleteFile('${vectorStoreId}', '${file.id}')" 
                                        class="btn-icon delete" title="Delete File">
                                    <i class="material-icons">delete</i>
                                </button>
                            </div>
                        </div>
                    `).join('')}
                </div>
                ${data.has_more ? `
                    <div class="load-more">
                        <button onclick="loadMoreFiles('${vectorStoreId}', '${data.last_id}')" class="btn btn-text">
                            Load More
                        </button>
                    </div>
                ` : ''}
            `;
        })
        .catch(error => {
            console.error('Error:', error);
            filesContainer.innerHTML = `
                <div class="loading">
                    <i class="material-icons" style="font-size: 48px; color: #d93025;">error_outline</i>
                    <div class="text-error">Error loading files: ${error.message}</div>
                </div>
            `;
        });
}

function loadMoreFiles(vectorStoreId, lastId) {
    const button = event.target;
    showLoader(button);
    
    loadVectorStoreFiles(vectorStoreId, lastId)
        .then(() => {
            hideLoader(button);
        })
        .catch(error => {
            hideLoader(button);
            showNotification('Error loading more files', 'error');
        });
}

function updateFileNames(input) {
    const fileList = Array.from(input.files)
        .map(file => file.name)
        .join(', ');
    input.closest('.form-group').querySelector('.selected-files').textContent = 
        fileList || 'No files chosen';
}

function formatFileSize(bytes) {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
}

function updateFilesList(input) {
    const filesContainer = document.getElementById('filesToUpload');
    const submitButton = document.querySelector('.submit-upload-btn');
    const files = Array.from(input.files);
    
    // Enable/disable upload button based on file selection
    submitButton.disabled = files.length === 0;
    
    if (files.length === 0) {
        filesContainer.innerHTML = '<div class="no-files">No files selected</div>';
        return;
    }
    
    filesContainer.innerHTML = files.map(file => `
        <div class="file-upload-item" data-file="${file.name}">
            <div class="file-upload-info">
                <div class="file-upload-name">${file.name}</div>
                <div class="file-upload-size">${formatFileSize(file.size)}</div>
            </div>
            <button type="button" class="btn-icon delete" onclick="removeFileFromUpload(this, '${file.name}')" title="Remove file">
                <i class="material-icons">close</i>
            </button>
        </div>
    `).join('');
}

function removeFileFromUpload(button, fileName) {
    const fileInput = document.getElementById('files');
    const fileItem = button.closest('.file-upload-item');
    
    // Create a new FileList without the removed file
    const dt = new DataTransfer();
    Array.from(fileInput.files)
        .filter(file => file.name !== fileName)
        .forEach(file => dt.items.add(file));
    
    fileInput.files = dt.files;
    
    // Remove the item from the list with animation
    fileItem.style.opacity = '0';
    fileItem.style.height = fileItem.offsetHeight + 'px';
    
    setTimeout(() => {
        fileItem.style.height = '0';
        fileItem.style.padding = '0';
        fileItem.style.margin = '0';
        
        setTimeout(() => {
            fileItem.remove();
            
            // If no files left, show "No files" message
            if (fileInput.files.length === 0) {
                document.getElementById('filesToUpload').innerHTML = 
                    '<div class="no-files">No files chosen</div>';
            }
        }, 300);
    }, 200);
}

async function uploadFiles(event) {
    event.preventDefault();
    const form = event.target;
    const fileInput = form.querySelector('input[type="file"]');
    const files = Array.from(fileInput.files);
    const vectorStoreId = document.getElementById('current_vectorstore_id').value;
    
    if (files.length === 0) return;

    const submitButton = form.querySelector('button[type="submit"]');
    submitButton.disabled = true;
    
    // Show spinner on button
    submitButton.innerHTML = `
        <div class="button-loader">
            <div class="loader"></div>
            <span>Uploading files...</span>
        </div>
    `;
    
    let completedFiles = 0;
    let failedFiles = 0;
    const BATCH_SIZE = 10;
    
    try {
        // Process files in batches of 10
        for (let i = 0; i < files.length; i += BATCH_SIZE) {
            const batch = files.slice(i, i + BATCH_SIZE);
            const formData = new FormData();
            
            batch.forEach(file => {
                formData.append('files[]', file);
            });

            try {
                const response = await fetch(`api/VectorStoresController.php?action=uploadFile&vectorstore_id=${vectorStoreId}`, {
                    method: 'POST',
                    body: formData
                });
                
                const data = await response.json();
                if (data.success) {
                    completedFiles += data.data.successful.length;
                    failedFiles += data.data.failed.length;
                    
                    // Reload files list after each successful batch
                    loadVectorStoreFiles(vectorStoreId);
                } else {
                    failedFiles += batch.length;
                }
            } catch (error) {
                console.error('Error processing batch:', error);
                failedFiles += batch.length;
            }
            
            // Small delay between batches
            await new Promise(resolve => setTimeout(resolve, 100));
        }
        
        // Show completion message
        if (failedFiles > 0) {
            showNotification(`Upload completed: ${completedFiles} successful, ${failedFiles} failed`, 'warning');
        } else {
            showNotification('All files uploaded successfully');
        }
        
    } catch (error) {
        console.error('Upload error:', error);
        showNotification('Upload failed: ' + error.message, 'error');
    } finally {
        submitButton.disabled = false;
        submitButton.innerHTML = '<i class="material-icons">cloud_upload</i> Upload Files';
        form.reset();
        closeFilesModal();
    }
}

function deleteFile(vectorStoreId, fileId) {
    if (confirm('Are you sure you want to delete this file?')) {
        const button = event.target.closest('button');
        const fileRow = button.closest('.file-item'); // Get the file row element
        showLoader(button);

        fetch('api/VectorStoresController.php?action=deleteFile', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({
                vectorstore_id: vectorStoreId,
                file_id: fileId
            })
        })
        .then(response => response.json())
        .then(data => {
            hideLoader(button);
            if (data.success) {
                // Animate and remove the row
                fileRow.style.transition = 'all 0.3s ease';
                fileRow.style.opacity = '0';
                fileRow.style.height = '0';
                fileRow.style.padding = '0';
                fileRow.style.margin = '0';
                
                setTimeout(() => {
                    fileRow.remove();
                    
                    // Check if there are no more files
                    const filesContainer = document.getElementById('vectorStoreFiles');
                    if (!filesContainer.querySelector('.file-item')) {
                        filesContainer.innerHTML = `
                            <div class="loading">
                                <i class="material-icons" style="font-size: 48px; color: var(--secondary-color);">folder_open</i>
                                <div>No files uploaded yet</div>
                            </div>
                        `;
                    }
                }, 300);
                
                showNotification('File deleted successfully');
            } else {
                showNotification('Error deleting file', 'error');
            }
        })
        .catch(error => {
            hideLoader(button);
            console.error('Error:', error);
            showNotification(error.message, 'error');
        });
    }
}

function updateFileName(input) {
    const fileName = input.files[0]?.name || 'No file chosen';
    input.closest('.form-group').querySelector('.selected-file').textContent = fileName;
}

function formatDate(timestamp) {
    return new Date(timestamp * 1000).toLocaleString();
} 