let currentPage = 1;
const itemsPerPage = 10;

const closeVectorStoreModal = () => {
    const modal = document.getElementById('vectorStoreModal');
    if (modal) {
        modal.style.display = 'none';
        modal.classList.remove('active');
    }
};

function saveVectorStore(event) {
    event.preventDefault();
    
    const form = event.target;
    const submitButton = form.querySelector('button[type="submit"]');
    showLoader(submitButton);
    
    const formData = new FormData(form);
    const vectorStoreId = formData.get('vectorstore_id');
    
    const data = {
        name: formData.get('name')
    };

    if (vectorStoreId) {
        data.vectorstore_id = vectorStoreId;
    }

    const action = vectorStoreId ? 'update' : 'create';
    
    fetch(`api/VectorStoresController.php?action=${action}`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json'
        },
        body: JSON.stringify(data)
    })
    .then(response => response.json())
    .then(data => {
        hideLoader(submitButton);
        if (data.success) {
            try {
                showNotification(vectorStoreId ? 'Vector Store updated successfully' : 'Vector Store created successfully');
                loadVectorStores(currentPage);
                closeVectorStoreModal();
            } catch (error) {
                console.error('Error after successful save:', error);
            }
        } else {
            showNotification('Error saving vector store', 'error');
        }
    })
    .catch(error => {
        hideLoader(submitButton);
        console.error('Error:', error);
        showNotification(error.message, 'error');
    });
} 

// Vector Stores functions
function loadVectorStores(page = 1) {
    currentPage = page;
    const tbody = document.getElementById('vectorstores-tbody');
    if (!tbody) {
        console.error('Vector stores tbody element not found');
        return;
    }

    console.log('Loading vector stores...');
    tbody.innerHTML = '<tr><td colspan="2" class="text-center">Loading...</td></tr>';

    fetch(`api/VectorStoresController.php?action=list&page=${page}`)
        .then(response => {
            console.log('Response status:', response.status);
            return response.json();
        })
        .then(data => {
            console.log('Vector stores data:', data);
            tbody.innerHTML = '';
            
            if (!data.data || data.data.length === 0) {
                tbody.innerHTML = '<tr><td colspan="2" class="text-center">No vector stores found</td></tr>';
                return;
            }
            
            data.data.forEach(store => {
                tbody.innerHTML += `
                    <tr>
                        <td>
                            <div>${store.name}</div>
                            <div class="store-id">${store.id}</div>
                        </td>
                        <td>
                            <button onclick="editVectorStore('${store.id}')" class="btn-icon" title="Edit">
                                <i class="material-icons">edit</i>
                            </button>
                            <button onclick="showFilesModal('${store.id}')" class="btn-icon" title="Files">
                                <i class="material-icons">folder</i>
                            </button>
                            <button onclick="deleteVectorStore('${store.id}')" class="btn-icon delete" title="Delete">
                                <i class="material-icons">delete</i>
                            </button>
                        </td>
                    </tr>
                `;
            });
            
            const pagination = document.getElementById('vectorstores-pagination');
            if (pagination) {
                let html = '';
                if (page > 1) {
                    html += `<button onclick="loadVectorStores(${page - 1})">Previous</button>`;
                }
                html += `<button class="active">${page}</button>`;
                if (data.has_more) {
                    html += `<button onclick="loadVectorStores(${page + 1})">Next</button>`;
                }
                pagination.innerHTML = html;
            }
        })
        .catch(error => {
            console.error('Error:', error);
            tbody.innerHTML = `
                <tr>
                    <td colspan="2" class="text-center text-error">
                        Error loading vector stores: ${error.message}
                    </td>
                </tr>
            `;
        });
}

// Initialize when page loads
document.addEventListener('DOMContentLoaded', () => {
    console.log('Checking for vector stores page...');
    if (window.location.href.includes('page=vectorstores')) {
        console.log('Loading vector stores page');
        loadVectorStores();

        // Add click handler for create button
        const createButton = document.getElementById('createVectorStoreBtn');
        if (createButton) {
            createButton.addEventListener('click', createVectorStore);
        }

        // Add click handlers for modal close buttons
        document.querySelectorAll('[data-action="close-modal"]').forEach(button => {
            button.addEventListener('click', () => {
                closeVectorStoreModal();
            });
        });
    }
});

// Add back the delete function
function deleteVectorStore(vectorStoreId) {
    if (confirm('Are you sure you want to delete this vector store?')) {
        const button = event.target.closest('button');
        const row = button.closest('tr'); // Get the parent row
        showLoader(button);

        fetch('api/VectorStoresController.php?action=delete', {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json'
            },
            body: JSON.stringify({ vectorstore_id: vectorStoreId })
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                showNotification('Vector store deleted successfully');
                
                // Animate and remove the row
                row.style.transition = 'all 0.3s ease';
                row.style.opacity = '0';
                row.style.height = '0';
                row.style.padding = '0';
                row.style.margin = '0';
                
                setTimeout(() => {
                    row.remove();
                    
                    // Check if there are no more vector stores
                    const tbody = document.getElementById('vectorstores-tbody');
                    if (!tbody.querySelector('tr')) {
                        tbody.innerHTML = '<tr><td colspan="2" class="text-center">No vector stores found</td></tr>';
                    }
                }, 300);
            } else {
                showNotification(data.error || 'Error deleting vector store', 'error');
            }
            hideLoader(button);
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Failed to delete vector store', 'error');
            hideLoader(button);
        });
    }
} 

// Add these functions to your existing vectorstores.js

function createVectorStore() {
    console.log('Creating vector store...');
    const modal = document.getElementById('vectorStoreModal');
    if (!modal) {
        console.error('Modal element not found');
        return;
    }

    const form = document.getElementById('vectorStoreForm');
    if (!form) {
        console.error('Form element not found');
        return;
    }

    // Reset form to its original state
    form.innerHTML = `
        <input type="hidden" id="vectorstore_id" name="vectorstore_id">
        
        <div class="form-group">
            <label for="name">Name</label>
            <input type="text" id="name" name="name" required>
        </div>

        <div class="form-actions">
            <button type="button" class="btn btn-secondary" onclick="closeVectorStoreModal()">Cancel</button>
            <button type="submit" class="btn btn-primary">Create</button>
        </div>
    `;
    
    form.reset();
    const vectorstoreIdInput = document.getElementById('vectorstore_id');
    if (vectorstoreIdInput) {
        vectorstoreIdInput.value = '';
    }

    const modalTitle = document.getElementById('modalTitle');
    if (modalTitle) {
        modalTitle.textContent = 'Create Vector Store';
    }
    
    // Show modal
    modal.style.display = 'flex';
    modal.classList.add('active');
}

function editVectorStore(vectorStoreId) {
    console.log('Editing vector store:', vectorStoreId);
    const modal = document.getElementById('vectorStoreModal');
    if (!modal) {
        console.error('Modal element not found');
        return;
    }

    const form = document.getElementById('vectorStoreForm');
    if (!form) {
        console.error('Form element not found');
        return;
    }

    const modalTitle = document.getElementById('modalTitle');
    if (modalTitle) {
        modalTitle.textContent = 'Edit Vector Store';
    }
    
    // Show loading state
    form.innerHTML = '<div class="loading">Loading...</div>';
    console.log('Showing modal...');
    showModal('vectorStoreModal');
    
    fetch(`api/VectorStoresController.php?action=retrieve&id=${vectorStoreId}`)
        .then(response => {
            if (!response.ok) throw new Error('Network response was not ok');
            return response.json();
        })
        .then(data => {
            if (data.error) throw new Error(data.error);
            
            // Restore form HTML
            form.innerHTML = `
                <input type="hidden" id="vectorstore_id" name="vectorstore_id">
                
                <div class="form-group">
                    <label for="name">Name</label>
                    <input type="text" id="name" name="name" required>
                </div>

                <div class="form-actions">
                    <button type="button" class="btn btn-secondary" onclick="closeVectorStoreModal()">Cancel</button>
                    <button type="submit" class="btn btn-primary">Save Changes</button>
                </div>
            `;
            
            // Set form values after ensuring elements exist
            const store = data.data;
            const idInput = document.getElementById('vectorstore_id');
            const nameInput = document.getElementById('name');
            
            if (idInput) idInput.value = store.id;
            if (nameInput) nameInput.value = store.name || '';
        })
        .catch(error => {
            console.error('Error:', error);
            showNotification('Error loading vector store: ' + error.message, 'error');
            closeVectorStoreModal();
        });
} 