<?php
require_once '../includes/config.php';

class AssignController {
    private $client;

    public function __construct($openAIClient) {
        $this->client = $openAIClient;
    }

    public function handleRequest() {
        try {
            header('Content-Type: application/json');
            
            $action = $_GET['action'] ?? 'list';
            
            $result = match($action) {
                'getAssignments' => $this->getAssignments(),
                'assign' => $this->assignVectorStores(),
                default => throw new \Exception('Action not found')
            };

            return json_encode($result);
        } catch (\Exception $e) {
            error_log('Controller Error: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => $e->getMessage()]);
        }
    }

    private function getAssignments() {
        try {
            $assistantId = $_GET['assistant_id'] ?? null;
            if (!$assistantId) {
                throw new \Exception('Assistant ID is required');
            }

            $assistant = $this->client->retrieveAssistant($assistantId);
            error_log('FULL Assistant Response: ' . json_encode($assistant, JSON_PRETTY_PRINT));
            
            // Extract vector store IDs from tool_resources
            $vectorStoreIds = [];
            
            // Check if file_search tool is enabled and tool_resources exists
            if (isset($assistant->tools)) {
                error_log('Tools found: ' . json_encode($assistant->tools));
                $hasFileSearch = false;
                foreach ($assistant->tools as $tool) {
                    if ($tool->type === 'file_search') {
                        $hasFileSearch = true;
                        break;
                    }
                }
                
                error_log('Has file_search: ' . ($hasFileSearch ? 'yes' : 'no'));
                error_log('Tool resources: ' . json_encode($assistant->tool_resources ?? null));
                
                if ($hasFileSearch && isset($assistant->tool_resources)) {
                    error_log('Checking tool_resources structure:');
                    error_log('- Has file_search: ' . (isset($assistant->tool_resources->file_search) ? 'yes' : 'no'));
                    if (isset($assistant->tool_resources->file_search)) {
                        error_log('- Has vector_store_ids: ' . (isset($assistant->tool_resources->file_search->vector_store_ids) ? 'yes' : 'no'));
                        if (isset($assistant->tool_resources->file_search->vector_store_ids)) {
                            $vectorStoreIds = $assistant->tool_resources->file_search->vector_store_ids;
                            error_log('Found vector store IDs: ' . json_encode($vectorStoreIds));
                        }
                    }
                }
            }

            $result = [
                'success' => true,
                'data' => $vectorStoreIds,
                'assistant' => [
                    'has_file_search' => $hasFileSearch ?? false,
                    'tools' => $assistant->tools ?? [],
                    'tool_resources' => $assistant->tool_resources ?? null
                ]
            ];
            
            error_log('Returning result: ' . json_encode($result, JSON_PRETTY_PRINT));
            return $result;
        } catch (\Exception $e) {
            error_log('Error getting assignments: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function assignVectorStores() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            error_log('Received assign request data: ' . json_encode($data, JSON_PRETTY_PRINT));
            
            if (!$data) {
                throw new \Exception('Invalid request data');
            }

            $assistantId = $data['assistant_id'] ?? null;
            $vectorStoreIds = $data['vector_store_ids'] ?? [];

            if (!$assistantId) {
                throw new \Exception('Assistant ID is required');
            }

            // Update assistant with vector store assignments
            $response = $this->client->associateVectorStoreWithAssistant($assistantId, $vectorStoreIds);
            error_log('OpenAI Response after assignment: ' . json_encode($response, JSON_PRETTY_PRINT));

            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error assigning vector stores: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }
}

// Handle the request
$controller = new AssignController($openAIClient);
echo $controller->handleRequest(); 