<?php
require_once '../includes/config.php';

class AssistantsController {
    private $client;

    public function __construct($openAIClient) {
        $this->client = $openAIClient;
    }

    public function handleRequest() {
        try {
            error_log("=== Starting AssistantsController handleRequest ===");
            error_log("Request Method: " . $_SERVER['REQUEST_METHOD']);
            error_log("Request URI: " . $_SERVER['REQUEST_URI']);
            error_log("Query String: " . $_SERVER['QUERY_STRING']);
            error_log("Action: " . ($_GET['action'] ?? 'none'));
            
            header('Content-Type: application/json');
            header('OpenAI-Beta: assistants=v2');
            
            $action = $_GET['action'] ?? 'list';
            
            $result = match($action) {
                'list' => $this->listAssistants(),
                'create' => $this->createAssistant(),
                'update' => $this->updateAssistant(),
                'delete' => $this->deleteAssistant(),
                'retrieve' => $this->retrieveAssistant(),
                default => throw new \Exception('Action not found')
            };

            error_log("Sending response: " . json_encode($result));
            return json_encode($result);
        } catch (\Exception $e) {
            error_log('Controller Error: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            http_response_code(500);
            return json_encode(['error' => $e->getMessage()]);
        }
    }

    private function listAssistants() {
        try {
            error_log("=== Starting listAssistants ===");
            error_log("Request URI: " . $_SERVER['REQUEST_URI']);
            error_log("Request Method: " . $_SERVER['REQUEST_METHOD']);
            
            $page = $_GET['page'] ?? 1;
            $limit = $_GET['limit'] ?? ITEMS_PER_PAGE;
            $after = $_GET['after'] ?? null;

            error_log("Params: page=$page, limit=$limit, after=$after");

            $response = $this->client->getAssistants($limit, $after);
            error_log('Assistants Raw Response: ' . print_r($response, true));
            
            $result = [
                'success' => true,
                'data' => $response->data,
                'has_more' => $response->hasMore
            ];
            
            error_log('Returning result: ' . print_r($result, true));
            return $result;
        } catch (\Exception $e) {
            error_log('Error in listAssistants: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function createAssistant() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $response = $this->client->createAssistant([
                'name' => $data['name'],
                'instructions' => $data['instructions'],
                'model' => $data['model'],
                'tools' => $data['tools'] ?? [],
                'file_ids' => $data['file_ids'] ?? []
            ]);
            return ['success' => true, 'data' => $response];
        } catch (Exception $e) {
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function updateAssistant() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            if (!$data) {
                throw new \Exception('Invalid request data');
            }

            $assistantId = $data['assistant_id'] ?? null;
            if (!$assistantId) {
                throw new \Exception('Assistant ID is required');
            }

            // Format tools array correctly
            $tools = array_map(function($tool) {
                return ['type' => $tool];
            }, $data['tools'] ?? []);

            error_log('Updating assistant: ' . $assistantId . ' with tools: ' . print_r($tools, true));

            // Remove assistant_id from data before sending to API
            unset($data['assistant_id']);

            $response = $this->client->updateAssistant($assistantId, [
                'name' => $data['name'],
                'instructions' => $data['instructions'],
                'model' => $data['model'],
                'tools' => $tools
            ]);

            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error updating assistant: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function deleteAssistant() {
        try {
            $assistantId = $_POST['assistant_id'];
            $response = $this->client->deleteAssistant($assistantId);
            return ['success' => true, 'data' => $response];
        } catch (Exception $e) {
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function retrieveAssistant() {
        try {
            $assistantId = $_GET['id'] ?? null;
            if (!$assistantId) {
                throw new \Exception('Assistant ID is required');
            }

            $response = $this->client->retrieveAssistant($assistantId);
            
            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error retrieving assistant: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }
}

// Error handling for the controller
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    throw new ErrorException($errstr, 0, $errno, $errfile, $errline);
});

try {
    $controller = new AssistantsController($openAIClient);
    echo $controller->handleRequest();
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
} 