<?php
require_once '../includes/config.php';
require_once '../includes/OpenAIClient.php';

// Add error logging
ini_set('display_errors', 1);
error_reporting(E_ALL);

header('Content-Type: application/json');

function logRequest() {
    $requestData = [
        'method' => $_SERVER['REQUEST_METHOD'],
        'get' => $_GET,
        'post' => $_POST,
    ];
    error_log("Request received: " . json_encode($requestData, JSON_PRETTY_PRINT));
}

logRequest();

function sendError($message, $code = 400) {
    $context = debug_backtrace(DEBUG_BACKTRACE_IGNORE_ARGS, 2)[0];
    error_log(sprintf(
        "Error occurred in %s:%d - %s",
        $context['file'],
        $context['line'],
        $message
    ));
    http_response_code($code);
    echo json_encode(['error' => $message]);
    exit;
}

function waitForRunCompletion($client, $threadId, $runId, $maxAttempts = 10) {
    $attempt = 0;
    $waitTime = 1;
    
    while ($attempt < $maxAttempts) {
        $runStatus = $client->threads()->runs()->retrieve(
            threadId: $threadId,
            runId: $runId
        );
        
        if ($runStatus['status'] === 'completed') {
            return true;
        }
        
        if (in_array($runStatus['status'], ['failed', 'cancelled', 'expired'])) {
            error_log("Run failed with status: " . $runStatus['status']);
            return false;
        }
        
        $attempt++;
        sleep($waitTime);
        $waitTime = min($waitTime * 2, 5); // Exponential backoff, max 5 seconds
    }
    
    return false;
}

function streamResponse($client, $threadId) {
    header('Content-Type: text/event-stream; charset=UTF-8');
    header('Cache-Control: no-cache');
    header('Connection: keep-alive');
    
    $messages = $client->threads()->messages()->list($threadId, [
        'limit' => 1,
        'order' => 'desc'
    ]);
    
    if (empty($messages['data'])) {
        return;
    }
    
    foreach ($messages['data'] as $message) {
        if ($message['role'] !== 'assistant') {
            continue;
        }
        
        foreach ($message['content'] as $content) {
            if ($content['type'] !== 'text') {
                continue;
            }
            
            $text = $content['text']['value'];
            $chunks = preg_split('/(?<=[\.\!\?\s])/u', $text, -1, PREG_SPLIT_NO_EMPTY);
            
            foreach ($chunks as $chunk) {
                $chunk = trim($chunk);
                if (empty($chunk)) {
                    continue;
                }
                
                echo json_encode(['content' => $chunk], 
                    JSON_UNESCAPED_UNICODE | JSON_PARTIAL_OUTPUT_ON_ERROR
                ) . "\n\n";
                
                ob_flush();
                flush();
                usleep(50000); // Reduced delay to 50ms
            }
        }
    }
    
    exit;
}

// Add this helper function at the top of the file, after the includes
function cleanMessageContent($content) {
    // Remove citation markers like 【26:6†source】
    return preg_replace('/【\d+:\d+†source】/', '', $content);
}

// Add these new helper functions after the existing ones
function processMessageContent($message, $fileNameMap = []) {
    $processedMsg = json_decode(json_encode($message), true);
    
    if ($processedMsg['role'] === 'assistant') {
        if (!empty($processedMsg['content'][0]['text']['value'])) {
            $processedMsg['content'][0]['text']['value'] = cleanMessageContent($processedMsg['content'][0]['text']['value']);
        }
        
        if (!empty($processedMsg['content'][0]['text']['annotations'])) {
            list($sources, $citations) = extractSourcesAndCitations(
                $processedMsg['content'][0]['text']['annotations'],
                $fileNameMap
            );
            
            if (!empty($sources)) {
                $processedMsg['sources'] = array_values(array_unique($sources));
                $processedMsg['citations'] = $citations;
            }
        }
    }
    
    return $processedMsg;
}

function extractSourcesAndCitations($annotations, $fileNameMap) {
    $sources = [];
    $citations = [];
    $processedFiles = [];
    
    foreach ($annotations as $annotation) {
        if ($annotation['type'] === 'file_citation' && 
            isset($annotation['file_citation']['file_id'])) {
            $fileId = $annotation['file_citation']['file_id'];
            
            if (isset($fileNameMap[$fileId])) {
                $fileName = $fileNameMap[$fileId];
                
                $citations[] = [
                    'file_name' => $fileName,
                    'file_id' => $fileId,
                    'quote' => $annotation['file_citation']['quote'] ?? '',
                    'text' => $annotation['text']
                ];
                
                if (!in_array($fileId, $processedFiles)) {
                    $sources[] = $fileName;
                    $processedFiles[] = $fileId;
                }
            }
        }
    }
    
    return [$sources, $citations];
}

// Add a caching mechanism for file name maps
function buildFileNameMap($steps) {
    static $cache = [];
    
    $cacheKey = md5(serialize($steps));
    if (isset($cache[$cacheKey])) {
        return $cache[$cacheKey];
    }
    
    $fileNameMap = [];
    
    if (isset($steps['data'])) {
        foreach ($steps['data'] as $step) {
            if ($step['type'] === 'tool_calls' && 
                isset($step['step_details']['tool_calls'])) {
                foreach ($step['step_details']['tool_calls'] as $toolCall) {
                    if (isset($toolCall['file_search']['results'])) {
                        foreach ($toolCall['file_search']['results'] as $result) {
                            if (isset($result['file_id']) && isset($result['file_name'])) {
                                $fileNameMap[$result['file_id']] = str_replace(
                                    '_', 
                                    ' ', 
                                    basename($result['file_name'], '.txt')
                                );
                            }
                        }
                    }
                }
            }
        }
    }
    
    $cache[$cacheKey] = $fileNameMap;
    return $fileNameMap;
}

try {
    // Initialize OpenAI client
    error_log("Initializing OpenAI client");
    $openAIClient = new OpenAIClient();
    $client = $openAIClient->getClient();
    $assistantId = $openAIClient->getAssistantId();
    
    // Check if this is a new thread request
    if (isset($_GET['vectorStoreId']) && !isset($_GET['threadId'])) {
        error_log("Creating new thread for vector store: " . $_GET['vectorStoreId']);
        // Create new thread
        $vectorStoreId = $_GET['vectorStoreId'];
        
        $threadResponse = $client->threads()->create([
            'tool_resources' => [
                'file_search' => [
                    'vector_store_ids' => [$vectorStoreId],
                ],
            ],
        ]);
        
        error_log("Thread created successfully: " . print_r($threadResponse, true));
        echo json_encode([
            'success' => true,
            'thread_id' => $threadResponse['id']
        ]);
        exit;
    }
    
    // Check if this is a message to existing thread
    if (isset($_GET['threadId']) && isset($_GET['messages'])) {
        $threadId = $_GET['threadId'];
        $message = $_GET['messages'];
        
        // Add user message to thread
        $client->threads()->messages()->create($threadId, [
            'role' => 'user',
            'content' => $message,
        ]);
        
        // Create and execute run
        error_log("Creating run with assistant ID: " . $assistantId);
        $runResponse = $client->threads()->runs()->create($threadId, [
            'assistant_id' => $assistantId,
            'tools' => [
                [
                    'type' => 'file_search',
                    'file_search' => [
                        'max_num_results' => 50,
                    ],
                ],
            ],
        ]);
        
        $runId = $runResponse['id'];
        
        // Wait for the run to complete
        if (waitForRunCompletion($client, $threadId, $runId)) {
            // Check if streaming is requested
            if (isset($_GET['stream']) && $_GET['stream'] === 'true') {
                streamResponse($client, $threadId);
            } else {
                // Regular JSON response
                $messages = $client->threads()->messages()->list($threadId, [
                    'limit' => 2,
                    'order' => 'desc'
                ]);
     
                // Get the run steps to find file names
                $stepsResponse = $client->threads()->runs()->steps()->list(
                    threadId: $threadId,
                    runId: $runId
                );
                
                // Use the new buildFileNameMap function
                $fileNameMap = buildFileNameMap($stepsResponse->toArray());
                
                $assistantResponse = null;
                foreach ($messages['data'] as $msg) {
                    if ($msg['role'] === 'assistant') {
                        // Use the new processMessageContent function
                        $assistantResponse = processMessageContent($msg, $fileNameMap);
                        break;
                    }
                }
  
                echo json_encode([
                    'success' => true,
                    'replay' => $assistantResponse ?? $messages
                ]);
            }
        } else {
            sendError('Run did not complete successfully', 500);
        }
        exit;
    }
    
    // In the try block, add this before the other conditions
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        $data = json_decode(file_get_contents('php://input'), true);
        
        if (isset($data['action']) && $data['action'] === 'delete' && isset($data['threadId'])) {
            try {
                // Delete thread from OpenAI
                $response = $client->threads()->delete($data['threadId']);
                
                // Check if deletion was successful
                if ($response->deleted) {
                    echo json_encode([
                        'success' => true,
                        'thread_id' => $response->id,
                        'object' => $response->object,
                        'deleted' => $response->deleted,
                        'response' => $response->toArray()
                    ]);
                    exit;
                } else {
                    sendError('Failed to delete thread from OpenAI');
                }
            } catch (Exception $e) {
                error_log("Error deleting thread from OpenAI: " . $e->getMessage());
                error_log("Stack trace: " . $e->getTraceAsString());
                sendError('Failed to delete thread: ' . $e->getMessage());
            }
        }
    }
    
    // Inside the try block, add this condition
    if (isset($_GET['action']) && $_GET['action'] === 'retrieve' && isset($_GET['threadId'])) {
        $threadId = $_GET['threadId'];
        
        // Get all messages for the thread
        $messages = $client->threads()->messages()->list($threadId, [
            'limit' => 100,
            'order' => 'asc'
        ]);
        
        // Get all runs for this thread
        $runs = $client->threads()->runs()->list($threadId);
        $messageRunMap = [];
        
        // Build file name map for each run
        if (!empty($runs['data'])) {
            foreach ($runs['data'] as $run) {
                $stepsResponse = $client->threads()->runs()->steps()->list(
                    threadId: $threadId,
                    runId: $run['id']
                );
                
                $messageRunMap[$run['id']] = buildFileNameMap($stepsResponse->toArray());
            }
        }
        
        // Process each message
        $processedMessages = [];
        if (isset($messages['data'])) {
            foreach ($messages['data'] as $msg) {
                $fileNameMap = [];
                if ($msg['role'] === 'assistant' && isset($msg['run_id'])) {
                    $fileNameMap = $messageRunMap[$msg['run_id']] ?? [];
                }
                
                $processedMessages[] = processMessageContent($msg, $fileNameMap);
            }
            
            echo json_encode([
                'success' => true,
                'messages' => [
                    'data' => $processedMessages,
                    'first_id' => $messages['first_id'],
                    'last_id' => $messages['last_id'],
                    'has_more' => $messages['has_more']
                ]
            ]);
            exit;
        }
    }
    
    sendError('Invalid request. Required parameters missing.');

} catch (Exception $e) {
    error_log("Exception caught: " . $e->getMessage());
    error_log("Stack trace: " . $e->getTraceAsString());
    sendError($e->getMessage(), 500);
} 