<?php
require_once '../includes/config.php';

class FilesController {
    private $client;

    public function __construct($openAIClient) {
        $this->client = $openAIClient;
    }

    public function handleRequest() {
        try {
            header('Content-Type: application/json');
            
            $action = $_GET['action'] ?? 'list';
            
            $result = match($action) {
                'list' => $this->listFiles(),
                'retrieve' => $this->retrieveFile(),
                'delete' => $this->deleteFile(),
                'deleteAll' => $this->deleteAll(),
                'upload' => $this->uploadFile(),
                default => throw new \Exception('Action not found')
            };

            return json_encode($result);
        } catch (\Exception $e) {
            error_log('Controller Error: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => $e->getMessage()]);
        }
    }

    private function listFiles() {
        try {
            $response = $this->client->files()->list();
            return [
                'success' => true,
                'data' => $response->data
            ];
        } catch (\Exception $e) {
            error_log('Error listing files: ' . $e->getMessage());
            throw $e;
        }
    }

    private function retrieveFile() {
        try {
            $fileId = $_GET['id'] ?? null;
            if (!$fileId) {
                throw new \Exception('File ID is required');
            }

            $response = $this->client->files()->retrieve($fileId);
            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error retrieving file: ' . $e->getMessage());
            throw $e;
        }
    }

    private function deleteFile() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $fileId = $data['file_id'] ?? null;
            
            if (!$fileId) {
                throw new \Exception('File ID is required');
            }

            $response = $this->client->files()->delete($fileId);
            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error deleting file: ' . $e->getMessage());
            throw $e;
        }
    }

    private function deleteAll() {
        try {
            // Get all files
            $response = $this->client->files()->list();
            $files = $response->data;
            
            $results = [
                'successful' => [],
                'failed' => []
            ];

            // Delete each file
            foreach ($files as $file) {
                try {
                    $deleteResponse = $this->client->files()->delete($file->id);
                    if ($deleteResponse->deleted) {
                        $results['successful'][] = $file->id;
                    } else {
                        $results['failed'][] = [
                            'id' => $file->id,
                            'error' => 'Delete operation failed'
                        ];
                    }
                } catch (\Exception $e) {
                    $results['failed'][] = [
                        'id' => $file->id,
                        'error' => $e->getMessage()
                    ];
                }
            }

            return [
                'success' => true,
                'data' => $results,
                'message' => count($results['successful']) . ' files deleted successfully'
            ];
        } catch (\Exception $e) {
            error_log('Error deleting all files: ' . $e->getMessage());
            throw $e;
        }
    }

    private function uploadFile() {
        try {
            if (!isset($_FILES['file'])) {
                throw new \Exception('No file uploaded');
            }

            $file = $_FILES['file'];
            $tempPath = $file['tmp_name'];
            
            // Upload to OpenAI
            $response = $this->client->files()->upload([
                'purpose' => 'assistants',
                'file' => fopen($tempPath, 'r')
            ]);

            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error uploading file: ' . $e->getMessage());
            return [
                'success' => false,
                'error' => $e->getMessage()
            ];
        }
    }
}

try {
    $controller = new FilesController($openAIClient);
    echo $controller->handleRequest();
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
} 