<?php
require_once '../includes/config.php';

class VectorStoresController {
    private $client;

    public function __construct($openAIClient) {
        $this->client = $openAIClient;
    }

    public function handleRequest() {
        try {
            header('Content-Type: application/json');
            header('OpenAI-Beta: assistants=v2');
            
            $action = $_GET['action'] ?? 'list';
            
            $result = match($action) {
                'list' => $this->listVectorStores(),
                'create' => $this->createVectorStore(),
                'update' => $this->updateVectorStore(),
                'delete' => $this->deleteVectorStore(),
                'retrieve' => $this->retrieveVectorStore(),
                'uploadFile' => $this->uploadFile(),
                'listFiles' => $this->listFiles(),
                'deleteFile' => $this->deleteFile(),
                default => throw new \Exception('Action not found')
            };

            return json_encode($result);
        } catch (\Exception $e) {
            error_log('Controller Error: ' . $e->getMessage());
            http_response_code(500);
            return json_encode(['error' => $e->getMessage()]);
        }
    }

    private function listVectorStores() {
        try {
            error_log("=== Starting listVectorStores ===");
            $page = $_GET['page'] ?? 1;
            $limit = $_GET['limit'] ?? ITEMS_PER_PAGE;
            $after = $_GET['after'] ?? null;

            error_log("Params: page=$page, limit=$limit, after=$after");

            $response = $this->client->getVectorStores($limit, $after);
            error_log('Vector Stores Raw Response: ' . print_r($response, true));
            
            $result = [
                'success' => true,
                'data' => $response->data,
                'has_more' => $response->hasMore
            ];
            
            error_log('Returning result: ' . print_r($result, true));
            return $result;
        } catch (\Exception $e) {
            error_log('Error in listVectorStores: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function createVectorStore() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            if (!$data) {
                throw new \Exception('Invalid request data');
            }

            $response = $this->client->createVectorStore([
                'name' => $data['name']
            ]);

            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error creating vector store: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function updateVectorStore() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            if (!$data) {
                throw new \Exception('Invalid request data');
            }

            $vectorStoreId = $data['vectorstore_id'] ?? null;
            if (!$vectorStoreId) {
                throw new \Exception('Vector Store ID is required');
            }

            $response = $this->client->updateVectorStore($vectorStoreId, [
                'name' => $data['name']
            ]);

            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error updating vector store: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function deleteVectorStore() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            $vectorStoreId = $data['vectorstore_id'] ?? null;
            if (!$vectorStoreId) {
                throw new \Exception('Vector Store ID is required');
            }

            $response = $this->client->deleteVectorStore($vectorStoreId);
            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error deleting vector store: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function retrieveVectorStore() {
        try {
            $vectorStoreId = $_GET['id'] ?? null;
            if (!$vectorStoreId) {
                throw new \Exception('Vector Store ID is required');
            }

            $response = $this->client->getVectorStore($vectorStoreId);
            
            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error retrieving vector store: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function uploadFile() {
        try {
            error_log("=== Starting file upload ===");
            
            $vectorStoreId = $_GET['vectorstore_id'] ?? null;
            if (!$vectorStoreId) {
                throw new \Exception('Vector Store ID is required');
            }

            if (empty($_FILES['files'])) {
                throw new \Exception('No files uploaded');
            }

            $results = [];
            $errors = [];

            foreach ($_FILES['files']['tmp_name'] as $key => $tmpName) {
                try {
                    $originalName = $_FILES['files']['name'][$key];
                    $tmpPath = $_FILES['files']['tmp_name'][$key];
                    $fileError = $_FILES['files']['error'][$key];
                    $fileSize = $_FILES['files']['size'][$key];

                    error_log("Processing file: $originalName");

                    if ($fileError !== UPLOAD_ERR_OK) {
                        throw new \Exception("Upload error code: $fileError");
                    }

                    if ($fileSize > MAX_UPLOAD_SIZE) {
                        throw new \Exception('File size exceeds limit of ' . (MAX_UPLOAD_SIZE / 1048576) . 'MB');
                    }

                    // Create a temporary file with original name
                    $tempDir = UPLOAD_DIR;
                    $tempPath = $tempDir . $originalName;

                    error_log("Moving uploaded file to: $tempPath");

                    if (!move_uploaded_file($tmpPath, $tempPath)) {
                        throw new \Exception("Failed to move uploaded file: $originalName");
                    }


                    // Upload to OpenAI
                    error_log("Uploading to OpenAI: $originalName");
                    $response = $this->client->uploadFileToVectorStore($vectorStoreId, $tempPath);

                    
                    // Delete the temporary file
                    if (file_exists($tempPath)) {
                        unlink($tempPath);
                        error_log("Deleted temporary file: $tempPath");
                    }

                    $results[] = [
                        'filename' => $originalName,
                        'success' => true,
                        'data' => $response
                    ];

                    error_log("Successfully processed file: $originalName");

                } catch (\Exception $e) {
                    error_log("Error processing file $originalName: " . $e->getMessage());
                    error_log($e->getTraceAsString());
                    
                    $errors[] = [
                        'filename' => $originalName,
                        'error' => $e->getMessage()
                    ];
                }
            }

            error_log("Upload process completed. Successful: " . count($results) . ", Failed: " . count($errors));

            return [
                'success' => true,
                'data' => [
                    'successful' => $results,
                    'failed' => $errors
                ]
            ];

        } catch (\Exception $e) {
            error_log("Fatal error in uploadFile: " . $e->getMessage());
            error_log($e->getTraceAsString());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }


    private function listFiles() {
        try {
            $vectorStoreId = $_GET['vectorstore_id'] ?? null;
            if (!$vectorStoreId) {
                throw new \Exception('Vector Store ID is required');
            }

            $limit = $_GET['limit'] ?? 10;
            $after = $_GET['after'] ?? null;

            $response = $this->client->getVectorStoreFiles($vectorStoreId, $limit, $after);
            
            // Get detailed information for each file
            $filesWithDetails = [];
            foreach ($response->data as $file) {
                try {
                    $fileDetails = $this->client->getFileDetails($file->id);
                    $filesWithDetails[] = [
                        'id' => $file->id,
                        'filename' => $fileDetails->filename,
                        'size' => $fileDetails->bytes,
                        'created_at' => $fileDetails->createdAt,
                        'status' => $fileDetails->status
                    ];
                } catch (\Exception $e) {
                    error_log('Error getting file details for ' . $file->id . ': ' . $e->getMessage());
                    $filesWithDetails[] = [
                        'id' => $file->id,
                        'filename' => 'Unknown',
                        'error' => true
                    ];
                }
            }
            
            return [
                'success' => true,
                'data' => $filesWithDetails,
                'has_more' => $response->hasMore,
                'last_id' => $response->lastId ?? null,
                'first_id' => $response->firstId ?? null
            ];
        } catch (\Exception $e) {
            error_log('Error listing files: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }

    private function deleteFile() {
        try {
            $data = json_decode(file_get_contents('php://input'), true);
            if (!$data) {
                throw new \Exception('Invalid request data');
            }

            $vectorStoreId = $data['vectorstore_id'] ?? null;
            $fileId = $data['file_id'] ?? null;

            if (!$vectorStoreId || !$fileId) {
                throw new \Exception('Vector Store ID and File ID are required');
            }

            $response = $this->client->deleteVectorStoreFile($vectorStoreId, $fileId);

            // Log both responses for debugging
            error_log('Vector Store Delete Response: ' . print_r($response['vector_store_response'], true));
            error_log('File Delete Response: ' . print_r($response['file_response'], true));

            // Check if both operations were successful
            $vectorStoreDeleted = $response['vector_store_response']->deleted ?? false;
            $fileDeleted = $response['file_response']->deleted ?? false;

            if (!$vectorStoreDeleted || !$fileDeleted) {
                throw new \Exception('Failed to delete file completely');
            }

            return [
                'success' => true,
                'data' => $response
            ];
        } catch (\Exception $e) {
            error_log('Error deleting file: ' . $e->getMessage());
            http_response_code(500);
            return ['error' => $e->getMessage()];
        }
    }
}

// Error handling for the controller
set_error_handler(function($errno, $errstr, $errfile, $errline) {
    throw new ErrorException($errstr, 0, $errno, $errfile, $errline);
});

try {
    $controller = new VectorStoresController($openAIClient);
    echo $controller->handleRequest();
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode(['error' => $e->getMessage()]);
} 