<?php
require_once __DIR__ . '/../vendor/autoload.php';

class OpenAIClient {
    private $client;
    private $assistantId;

    public function __construct() {
        $this->client = OpenAI::client(OPENAI_API_KEY);
        $this->assistantId = OPENAI_ASSISTANT_ID;
    }

    public function getClient() {
        return $this->client;
    }

    public function getAssistantId() {
        return $this->assistantId;
    }

    // Assistants CRUD
    public function getAssistants($limit = 10, $after = null) {
        try {
            error_log("Attempting to fetch assistants with limit: $limit, after: $after");
            $response = $this->client->assistants()->list([
                'limit' => $limit,
                'after' => $after
            ], [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
            error_log('OpenAI Response: ' . print_r($response, true));
            return $response;
        } catch (\Exception $e) {
            error_log('OpenAI Error: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    public function createAssistant($params) {
        return $this->client->assistants()->create([
            'name' => $params['name'],
            'instructions' => $params['instructions'],
            'model' => $params['model'],
            'tools' => $params['tools'] ?? [],
            'file_ids' => $params['file_ids'] ?? []
        ], [
            'headers' => [
                'OpenAI-Beta' => 'assistants=v2'
            ]
        ]);
    }

    public function updateAssistant($assistantId, $params) {
        try {
            error_log('Updating assistant with params: ' . print_r($params, true));
            
            return $this->client->assistants()->modify($assistantId, [
                'name' => $params['name'],
                'instructions' => $params['instructions'],
                'model' => $params['model'],
                'tools' => $params['tools']
            ], [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
        } catch (\Exception $e) {
            error_log('Error in updateAssistant: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    public function deleteAssistant($assistantId) {
        return $this->client->assistants()->delete($assistantId, [
            'headers' => [
                'OpenAI-Beta' => 'assistants=v2'
            ]
        ]);
    }

    // Vector Stores CRUD
    public function getVectorStores($limit = 10, $after = null) {
        try {
            error_log("Attempting to fetch vector stores with limit: $limit, after: $after");
            $response = $this->client->vectorStores()->list([
                'limit' => $limit,
                'after' => $after
            ], [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
            error_log('Vector Stores Response: ' . print_r($response, true));
            return $response;
        } catch (\Exception $e) {
            error_log('OpenAI Error: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    public function createVectorStore($params) {
        try {
            return $this->client->vectorStores()->create([
                'name' => $params['name']
            ], [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
        } catch (\Exception $e) {
            error_log('Error creating vector store: ' . $e->getMessage());
            throw $e;
        }
    }

    public function updateVectorStore($vectorStoreId, $params) {
        try {
            error_log('Updating vector store with params: ' . print_r($params, true));
            
            return $this->client->vectorStores()->modify($vectorStoreId, [
                'name' => $params['name']
            ], [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
        } catch (\Exception $e) {
            error_log('Error updating vector store: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    public function deleteVectorStore($vectorStoreId) {
        try {
            return $this->client->vectorStores()->delete($vectorStoreId, [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
        } catch (\Exception $e) {
            error_log('Error deleting vector store: ' . $e->getMessage());
            throw $e;
        }
    }

    public function getVectorStore($vectorStoreId) {
        try {
            return $this->client->vectorStores()->retrieve($vectorStoreId, [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
        } catch (\Exception $e) {
            error_log('Error retrieving vector store: ' . $e->getMessage());
            throw $e;
        }
    }

    // Vector Store Files
    public function uploadFileToVectorStore($vectorStoreId, $filePath) {
        try {
            error_log("Uploading file to OpenAI: $filePath");

            // First upload file to OpenAI
            $fileResponse = $this->client->files()->upload([
                'purpose' => 'assistants',
                'file' => fopen($filePath, 'r')
            ]);
            
           
            error_log("File uploaded to OpenAI. File ID: " . $fileResponse->id);

            // Then associate it with the vector store
            $response = $this->client->vectorStores()->files()->create(
                vectorStoreId: $vectorStoreId,
                parameters: [
                    'file_id' => $fileResponse->id,
                ]
            );
            
            error_log("File associated with vector store. Response: " . print_r($response, true));
            
            return $response;
        } catch (\Exception $e) {
            error_log("Error in uploadFileToVectorStore: " . $e->getMessage());
            error_log($e->getTraceAsString());
            throw $e;
        }
    }

    public function getVectorStoreFiles($vectorStoreId, $limit = 10, $after = null) {
        return $this->client->vectorStores()->files()->list(
            vectorStoreId: $vectorStoreId,
            parameters: [
                'limit' => $limit,
                'after' => $after
            ]
        );
    }

    public function deleteVectorStoreFile($vectorStoreId, $fileId) {
        try {
            // First, remove the file from the vector store
            $response = $this->client->vectorStores()->files()->delete(
                vectorStoreId: $vectorStoreId,
                fileId: $fileId
            );

            // Then, delete the actual file
            $fileResponse = $this->client->files()->delete($fileId);

            return [
                'vector_store_response' => $response,
                'file_response' => $fileResponse
            ];
        } catch (\Exception $e) {
            error_log('Error deleting vector store file: ' . $e->getMessage());
            throw $e;
        }
    }

    // Vector Store File Batches
    public function createFileBatch($vectorStoreId, $params) {
        return $this->client->vectorStores()->fileBatches()->create(
            vectorStoreId: $vectorStoreId,
            parameters: [
                'file_ids' => $params['file_ids'],
                'purpose' => $params['purpose'] ?? 'assistants'
            ]
        );
    }

    public function getFileBatches($vectorStoreId, $limit = 10, $after = null) {
        return $this->client->vectorStores()->fileBatches()->list(
            vectorStoreId: $vectorStoreId,
            parameters: [
                'limit' => $limit,
                'after' => $after
            ]
        );
    }

    public function getFileBatch($vectorStoreId, $batchId) {
        return $this->client->vectorStores()->fileBatches()->retrieve(
            vectorStoreId: $vectorStoreId,
            fileBatchId: $batchId
        );
    }

    public function retrieveAssistant($assistantId) {
        return $this->client->assistants()->retrieve($assistantId, [
            'headers' => [
                'OpenAI-Beta' => 'assistants=v2'
            ]
        ]);
    }

    public function associateVectorStoreWithAssistant($assistantId, $vectorStoreIds) {
        try {
            // Take only the first vector store ID if multiple are provided
            $vectorStoreId = !empty($vectorStoreIds) ? [$vectorStoreIds[0]] : [];
            error_log('Associating vector store. Assistant ID: ' . $assistantId . ', Vector Store ID: ' . print_r($vectorStoreId, true));
            
            $params = [
                'tools' => [['type' => 'file_search']],
                'tool_resources' => [
                    'file_search' => [
                        'vector_store_ids' => $vectorStoreId // Single vector store ID in array
                    ],
                ]
            ];
            
            error_log('Sending params to OpenAI: ' . print_r($params, true));
            
            return $this->client->assistants()->modify($assistantId, $params, [
                'headers' => [
                    'OpenAI-Beta' => 'assistants=v2'
                ]
            ]);
        } catch (\Exception $e) {
            error_log('Error associating vector store: ' . $e->getMessage());
            error_log('Stack trace: ' . $e->getTraceAsString());
            throw $e;
        }
    }

    public function getFileDetails($fileId) {
        try {
            return $this->client->files()->retrieve($fileId);
        } catch (\Exception $e) {
            error_log('Error retrieving file details: ' . $e->getMessage());
            throw $e;
        }
    }

    // Add this method to the OpenAIClient class
    public function files() {
        return new class($this->client) {
            private $client;

            public function __construct($client) {
                $this->client = $client;
            }

            public function list() {
                try {
                    error_log("Listing files...");
                    return $this->client->files()->list();
                } catch (\Exception $e) {
                    error_log("Error listing files: " . $e->getMessage());
                    throw $e;
                }
            }

            public function retrieve($fileId) {
                try {
                    error_log("Retrieving file: " . $fileId);
                    return $this->client->files()->retrieve($fileId);
                } catch (\Exception $e) {
                    error_log("Error retrieving file: " . $e->getMessage());
                    throw $e;
                }
            }

            public function delete($fileId) {
                try {
                    error_log("Deleting file: " . $fileId);
                    return $this->client->files()->delete($fileId);
                } catch (\Exception $e) {
                    error_log("Error deleting file: " . $e->getMessage());
                    throw $e;
                }
            }

            public function upload($params) {
                try {
                    error_log("Uploading file with params: " . print_r($params, true));
                    return $this->client->files()->upload($params);
                } catch (\Exception $e) {
                    error_log("Error uploading file: " . $e->getMessage());
                    throw $e;
                }
            }
        };
    }
} 